/********************************************************************************

   cpuclock.cpp
   
   Copyright 2007-2024 Michael Cornelison
   source code URL: https://kornelix.net
   contact: mkornelix@gmail.com

   This is a utility program to adjust the CPU clock speed and
   evaluate the effect on CPU temperature and system power draw.
   This program works for some recent Intel CPUs. 

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version. See https://www.gnu.org/licenses

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
   See the GNU General Public License for more details.

*********************************************************************************/

#include "zfuncs.h"

#define release  "cpuclock-3.1"                                                        //  3.1
#define appname  "cpuclock"
#define website  "https://kornelix.net/cpuclock/cpuclock.html"
#define contact  "mkornelix@gmail.com"

#define freqfolder "/sys/devices/system/cpu/cpufreq/policy1/"                          //  this could change


zdialog     *zdmain;                                                                   //  main window
GtkWidget   *mwin = 0;
int         cpuload = 0;                                                               //  start or stop cpu load
int         cpubusy = 0;                                                               //  cpu thread busy count
int         threads = 0;                                                               //  thread count
float       minclock, maxclock, currclock;                                             //  CPU clock, GHz units
int         perf100;                                                                   //  iterations for 1 thread, 1 second
int         iterations;                                                                //  iterations for all threads
int         Fcalibrated = 0;                                                           //  calibration done
double      sampleT1 = 0, sampleT2;                                                    //  performance sample interval

namespace zfuncs {                                                                     //  externals from zfuncs module
   extern GdkDisplay        *display;                                                  //  X11 workstation (KB, mouse, screen)
   extern GdkDeviceManager  *manager;                                                  //  knows screen / mouse associations
   extern GdkScreen         *screen;                                                   //  monitor (screen)
   extern GdkDevice         *mouse;                                                    //  pointer device
   extern GtkSettings       *settings;                                                 //  screen settings
   extern GtkWidget         *mainwin;                                                  //  main window for zfuncs parent
}


//  main program

int main(int argc, ch *argv[])
{
   int   zdmain_event(zdialog *zd, ch *event);
   int   get_hardware_data(void *);
   int   get_cpu_clock_range(void *);

   int      err;
   int      Fcpupower, Fpowerstat;
   ch       clockrange[60];                                                            //  "CPU clock range: N.N to N.N GHz"
   ch       clockset[40];                                                              //  spin button range, min|max|step|curr
   ch       message[100];
   
   zinitapp(release,argc,argv);                                                        //  get app directories etc.

   if (geteuid() != 0) {
      zmessageACK(mwin,"cpuclock must be run with sudo");                              //  2.3
      exit(0);
   }
   
   Fcpupower = Fpowerstat = 0;                                                         //  check required programs are installed

   err = system("which cpupower >/dev/null 2>&1");
   if (err) Fcpupower = 1;

   err = system("which powerstat >/dev/null 2>&1");
   if (err) Fpowerstat = 1;

   if (Fcpupower || Fpowerstat) {
      strcpy(message,"please install: \n");
      if (Fcpupower) strcat(message," linux-tools-generic \n");
      if (Fpowerstat) strcat(message," powerstat \n");
      zmessageACK(0,message);
      exit(0);
   }

   if (argc > 1)                                                                       //  command line: $ sudo cpuclock N.N
   {
      zshell("log","cpupower frequency-set -u %sGHz >/dev/null",argv[1]);
      exit(0);
   }

   get_cpu_clock_range(0);                                                             //  get CPU clock data: min, max, current
   snprintf(clockrange,40,"CPU clock range: %.1f - %.1f GHz",minclock,maxclock);
   snprintf(clockset,40,"%.1f|%.1f|0.1|%.1f",minclock, maxclock, currclock);

/***
       ___________________________________
      |           Set CPU Clock           |
      |                                   |
      |  CPU clock range: N.N - N.N GHz   |     label    clockrange
      |  [ +- 4.0 ] GHz   [ set ]         |     spin     GHz            button   clockset
      |  Sample number: NNN               |     label    sample
      |  Current CPU power: 112 Watts     |     label    watts
      |  Current CPU temp: 55°C           |     label    temp
      |  Compute threads: [ +- 4 ]        |     spin     threads
      |  Actual performance: NN.N %       |     label    perf
      |                                   |
      |                     [Help] [Quit] |
      |___________________________________|

***/

   zdmain = zdialog_new("Set CPU Clock",0,"Help","Quit",0);                            //  create main window
   mwin = zdmain->dialog;                                                              //  GtkWidget *
   zfuncs::mainwin = mwin;
   
   zdialog_add_widget(zdmain,"hbox","hbrange","dialog",0,"space=3");
   zdialog_add_widget(zdmain,"label","clockrange","hbrange",clockrange,"space=3");

   zdialog_add_widget(zdmain,"hbox","hbclock","dialog",0,"space=3");
   zdialog_add_widget(zdmain,"spin","GHz","hbclock",clockset,"space=3|size=4");
   zdialog_add_widget(zdmain,"label","labGHz","hbclock","GHz","space=8");
   zdialog_add_widget(zdmain,"button","clockset","hbclock","set","space=3");
   
   zdialog_add_widget(zdmain,"hbox","hbsamp","dialog",0,"space=3");
   zdialog_add_widget(zdmain,"label","sample","hbsamp","Sample number:  0","space=3"); 

   zdialog_add_widget(zdmain,"hbox","hbwatts","dialog",0,"space=3");
   zdialog_add_widget(zdmain,"label","watts","hbwatts","Current CPU power:  0.00 Watts","space=3");

   zdialog_add_widget(zdmain,"hbox","hbtemp","dialog",0,"space=3");
   zdialog_add_widget(zdmain,"label","temp","hbtemp","Current CPU temp:  00 °C","space=3");
   
   zdialog_add_widget(zdmain,"hbox","hbth","dialog",0,"space=3");
   zdialog_add_widget(zdmain,"label","labth","hbth","Compute threads:","space=3");
   zdialog_add_widget(zdmain,"spin","threads","hbth","0|99|1|0","space=3|size=4");

   zdialog_add_widget(zdmain,"hbox","hbperf","dialog",0,"space=3");
   zdialog_add_widget(zdmain,"label","perf","hbperf","Actual performance: ","space=3");

   zdialog_stuff(zdmain,"GHz",currclock);

   zdialog_resize(zdmain,300,0);
   zdialog_run(zdmain,zdmain_event);                                                   //  show main window
   zmainsleep(1);                                                                      //  let it paint

   g_timeout_add(100,get_hardware_data,0);                                             //  start periodic function
   
   zdialog_send_event(zdmain,"calibrate");                                             //  do calibration

   zdialog_wait(zdmain);
   zdialog_free(zdmain);
   exit(0);
}


//  main window dialog event and completion function

int zdmain_event(zdialog *zd, ch *event)
{
   void m_help();
   void cpu_load();
   void set_cpu_clock();
   
   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                            //  [Help]
         zd->zstat = 0;
         m_help();
         return 1;
      }
      else {                                                                           //  [quit] or [x]
         zshell("log","pkill powerstat >/dev/null");
         exit(0);
      }
   }
   
   if (strmatch(event,"focus")) return 1;
   
   if (strmatch(event,"clockset"))                                                     //  change CPU nominal clock speed
   {
      set_cpu_clock();
      Fcalibrated = 0;                                                                 //  needs new calibration
   }
      
   if (strmatch(event,"threads"))                                                      //  change thread count
   {
      cpuload = 0;                                                                     //  stop threads
      while (cpubusy) zsleep(0.1);                                                     //  wait for stop
      zdialog_fetch(zd,"threads",threads);                                             //  get new thread count
      cpu_load();                                                                      //  start with new thread count
   }

   if (! Fcalibrated)                                                                  //  calibrate using 1 thread
   {
      zdialog_stuff(zdmain,"threads",0);

      cpuload = 0;                                                                     //  stop threads
      while (cpubusy) zsleep(0.1);                                                     //  wait for stop

      threads = 1;                                                                     //  run 1 thread for 1 second
      cpuload = 1;
      iterations = 0;
      cpu_load();
      zsleep(1.0);

      cpuload = 0;                                                                     //  stop threads
      while (cpubusy) zsleep(0.1);                                                     //  wait for stop

      perf100 = iterations;                                                            //  values vary about 2% 
      printf("perf100: %d \n",perf100);                                                //  4.9 GHz -> 30400
      Fcalibrated = 1;
      sampleT1 = 0;                                                                    //  start new sample period

      zdialog_stuff(zdmain,"threads",0);
      threads = 0;
   }

   return 1;
}


//  show help menu

void m_help()
{
   void show_userguide(GtkWidget *, ch *menu);
   void show_about(GtkWidget *, ch *menu);

   GtkWidget   *menu;

   menu = create_popmenu();
   add_popmenu(menu,"user guide",show_userguide,0,0);
   add_popmenu(menu,"about cpuclock",show_about,0,0);
   popup_menu(mwin,menu);

   return;
}


//  show the user guide in a popup window

void show_userguide(GtkWidget *, ch *menu)
{
   showz_docfile(0,"userguide",0);
   return;
}


//  show the about text in a popup window

void show_about(GtkWidget *, ch *menu)
{
   zabout(mwin);
   return;
}


//  start 'threads' processes to load the CPU

void cpu_load()
{
   void * cpu_load_threadfunc(void *);
   
   if (! threads) {
      cpuload = 0;
      cpubusy = 0;
      return;
   }

   cpuload = 1;

   for (int ii = 0; ii < threads; ii++)
      start_detached_thread(cpu_load_threadfunc,0);

   cpubusy = threads;                                                                  //  incr. busy threads
   return;   
}


//  do calculations for one thread and count iterations

void * cpu_load_threadfunc(void *)
{
   float load_function(void);
   
   while (true)
   {
      if (! cpuload) {                                                                 //  no load, exit thread
         zadd_locked(cpubusy,-1);                                                      //  decr. busy threads
         pthread_exit(0);
      }

      load_function();                                                                 //  do CPU load function

      zadd_locked(iterations,+1);                                                      //  count iterations
   }
}


float load_function(void)
{
   float  fval = drandz();
   float  data[200];
   int    ii, jj;

   for (ii = 0; ii < 200; ii++) 
      data[ii] = 5.0 * drandz();                                                       //  0 - 4.999... 
   
   for (ii = 0; ii < 200; ii++) 
   {
      for (jj = 0; jj < 199; jj++)
      {
         fval = data[jj] * data[jj+1];
         fval = sinf(fval/2.0);
         fval = cosf(fval);
         fval = sqrtf(fval);
      }
   }
   
   return fval;
}


//  get CPU clock allowed range and current max. frequency

int get_cpu_clock_range(void *) 
{
   FILE     *fid = 0;
   ch       *pp, buff[200];
   float    min = 0, max = 0, curr = 0;
   
   fid = popen("cat " freqfolder "cpuinfo_max_freq","r");
   if (fid) {
      pp = fgets(buff,100,fid);
      if (pp) max = atof(pp);
      pclose(fid);
   }
   
   fid = popen("cat " freqfolder "cpuinfo_min_freq","r");
   if (fid) {
      pp = fgets(buff,100,fid);
      if (pp) min = atof(pp);
      pclose(fid);
   }
   
   fid = popen("cat " freqfolder "scaling_max_freq","r");
   if (fid) {
      pp = fgets(buff,100,fid);
      if (pp) curr = atof(pp);
      pclose(fid);
   }
   
   if (! min) min = 800000;                                                            //  units are KHz
   if (! max) max = 3000000;
   if (! curr) curr = max;
   
   min = 0.000001 * min;                                                               //  convert to GHz
   max = 0.000001 * max;
   curr = 0.000001 * curr;

   minclock = min;
   maxclock = max;
   currclock = curr;

   return 1;
}


//  set max. CPU clock speed

void set_cpu_clock()
{
   int      err;
   float    GHz;
   ch       command[100];
   
   zdialog_fetch(zdmain,"GHz",GHz);
   snprintf(command,100,"cpupower frequency-set -u %.1fGHz >/dev/null",GHz);
   err = zshell("log",command);
   if (err) {
      zmessageACK(mwin,"cpupower command failed - check log file");
      exit(0);
   }
   printf("CPU clock set to %.1f GHz \n",GHz);
   return;
}


//  periodic function to get CPU temperature and power and update widget values

int get_hardware_data(void *)
{
   FILE        *fid = 0;
   int         ii, err, posn = -1, temp;
   float       watts, wattsum = 0, wattscount = 0;
   ch          *pp, command[200], buff[200];
   ch          wattsfile[200];
   static int  sample = 0, avgtemp = 0;

   //  start powerstat process to collect multiple power draw samples

   snprintf(wattsfile,200,"%s/wattsdata",get_zhomedir());                              //  file for watts data
   remove(wattsfile);

   snprintf(command,200,"powerstat -R >%s &",wattsfile);
   err = system(command);                                                              //  watts data >> wattsfile
   if (err) goto powerstat_err;

   //  get 3 temperature samples, compute running weighted average

   for (ii = 0; ii < 3; ii++)
   {
      temp = coretemp();                                                               //  get CPU temp.
      if (avgtemp == 0) avgtemp = temp;                                                //  1st sample
      avgtemp = 0.6 * avgtemp + 0.4 * temp + 0.5;                                      //  recent samples, weighted average
      zmainsleep(1);                                                                   //  delay 1 sec.
   }
   
   snprintf(buff,200,"Current CPU temp:  %2d °C",avgtemp);                             //  update CPU temp. in main window
   zdialog_stuff(zdmain,"temp",buff);
   zmainloop();
   
   err = system("pkill powerstat >/dev/null 2>&1");                                    //  kill powerstat process
   err = system("pkill powerstat >/dev/null 2>&1");                                    //  wait until dead - necessary
   
   fid = fopen(wattsfile,"r");                                                         //  open wattsfile
   if (! fid) goto powerstat_err;

   while (true)                                                                        //  read output recs
   {
      pp = fgets(buff,200,fid);
      if (! pp) goto powerstat_err;                                                    //  no header found
      pp = strstr(buff," Watts");                                                      //  look for header record
      if (! pp) continue;
      posn = pp - buff;                                                                //  " Watts" header position
      break;
   }

   if (posn < 0) goto powerstat_err;                                                   //  no header found

   while (true)                                                                        //  read output recs
   {
      pp = fgets(buff,200,fid);
      if (! pp) break;
      if (strstr(pp,"----")) break;                                                    //  end of data recs
      watts = atof(pp+posn);                                                           //  watts sample
      if (watts < 0 || watts > 900) continue;                                          //  format?
      wattsum += watts;                                                                //  watts sum
      wattscount++;                                                                    //  count samples
   }
   
   fclose(fid);
   fid = 0;
   
   if (wattscount == 0) goto powerstat_err;                                            //  no data found
   watts = wattsum / wattscount;                                                       //  average watts

   snprintf(buff,200,"Current CPU power:  %.1f Watts",watts);                          //  update watts in main window
   zdialog_stuff(zdmain,"watts",buff);

   sample++;
   snprintf(buff,200,"Sample number:  %d",sample);                                     //  update sample no. in main window
   zdialog_stuff(zdmain,"sample",buff);
   
//  measure actual performance = iterations / elapsed_time / perf100;

   float    perfact, elapsed;
   ch       perftext[100];

   if (! sampleT1) {                                                                   //  new sample period
      sampleT1 = get_seconds();                                                        //  set period start time
      return 1;
   }
   
   if (! Fcalibrated || ! threads) {
      zdialog_stuff(zdmain,"perf","Actual performance: ");
      return 1;
   }

   sampleT2 = get_seconds();                                                           //  sample done, get period end time
   elapsed = sampleT2 - sampleT1;                                                      //  sample period, seconds
   sampleT1 = sampleT2;                                                                //  next period start time

   perfact = iterations / elapsed / threads / perf100 * 100;                           //  performance percent

   zput_locked(iterations,0);                                                          //  reset computations counter
   
   snprintf(perftext,100,"Actual performance: %.1f",perfact);                          //  update dialog window
   zdialog_stuff(zdmain,"perf",perftext);

   return 1;

powerstat_err:
   zmessageACK(mwin,"powerstat -R command failed");
   if (fid) fclose(fid);
   exit(0);
}


//  supply unused zdialog callback function

void KBevent(GdkEventKey *event) { return; }



